﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Domain.Services;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.UserModels.Filters;
    using Utilities;
    using Hims.Shared.Library.Enums;
    using Hims.Api.Models;

    /// <inheritdoc />
    [AllowAnonymous]
    [Route("api/radiology-value")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class RadiologyValueController : BaseController
    {
        /// <summary>
        /// The iRadiologyValueService services.
        /// </summary>
        private readonly IRadiologyValueService radiologyValueService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public RadiologyValueController(IRadiologyValueService radiologyValueService, IAuditLogService auditLogServices) { this.radiologyValueService = radiologyValueService; this.auditLogServices = auditLogServices; }

        /// <summary>
        /// The fetch RadiologyValue.
        /// </summary>
        /// <param name="model">
        /// The RadiologyValue filter model.
        /// </param>
        /// <returns>
        /// The list of RadiologyValue.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of RadiologyValues.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<RadiologyValueModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody] RadiologyValueFilterModel model)
        {
            model = (RadiologyValueFilterModel)EmptyFilter.Handler(model);
            var radiologyValues = await this.radiologyValueService.FetchAsync(model);
            return this.Success(radiologyValues);
        }

        /// <summary>
        /// The add RadiologyValue.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - RadiologyValue added successfully.
        /// - 409 - RadiologyValue already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody] RadiologyValueModel model, [FromHeader] LocationHeader header)
        {
            model = (RadiologyValueModel)EmptyFilter.Handler(model);
            var response = await this.radiologyValueService.AddAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given RadiologyValue name has already been exists with us.");
                case 0:
                    return this.ServerError();
            }
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.RadiologyValues,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow,
                LogDescription = $"{model.Name} RadiologyValue has been added.",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return this.Success("RadiologyValue has been added successfully.");
        }

        /// <summary>
        /// The update RadiologyValue.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - RadiologyValue updated successfully.
        /// - 409 - RadiologyValue already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] RadiologyValueModel model, [FromHeader] LocationHeader header)
        {
            model = (RadiologyValueModel)EmptyFilter.Handler(model);
            var response = await this.radiologyValueService.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given RadiologyValue name has already been exists with us.");
                case 0:
                    return this.ServerError();
            }
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.RadiologyValues,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow,
                LogDescription = $"{model.Name} RadiologyValue has been updated successfully.",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return this.Success("RadiologyValue has been updated successfully.");
        }

        /// <summary>
        /// The delete RadiologyValue.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - RadiologyValue deleted successfully.
        /// - 409 - RadiologyValue can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody] RadiologyValueModel model,[FromHeader] LocationHeader header)
        {
            try
            {
                model = (RadiologyValueModel)EmptyFilter.Handler(model);
                var response = await this.radiologyValueService.DeleteAsync(model.RadiologyValueId);
                var responseProviderEncounter = await this.radiologyValueService.updateProviderEncounterAsync(model.RadiologyValueId);
                if (response == 0)
                {
                    return this.ServerError();
                }
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.RadiologyValues,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.UtcNow,
                    LogDescription = $"{model.Name} RadiologyValue has been deleted successfully.",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);
                return this.Success("RadiologyValue has been deleted successfully.");
            }
            catch (NpgsqlException exception)
            {
                return this.ServerError(exception.Message);
            }
        }
    }
}